// HUD Logs 1.0
// October 2018
// by ZylonBane
// Displays the log portrait, sender, and subject of emails in the HUD when they
// arrive. Optionally automatically plays logs when they're picked up.
//
// Overrides OSM scripts LogDiscScript, TrapLog, TrapEmail, and EmailRoom.

// --------------------------------------------------------------------------------
// Script configuration
const AUTO_LOGS  = true;  // automatically play logs when picked up
const POPUP_TIME = 10;    // seconds to display log info popup

// --------------------------------------------------------------------------------
// Other globals
local _logBitmap;
local _logString;
const HUD_CON    = "zbHudCon";
const TYPE_EMAIL = 0;
const TYPE_LOG   = 1;
const S_OK       = 0;
const S_FALSE    = 1;

// --------------------------------------------------------------------------------
// Base class for displaying logs/emails with HUD icons
// log types: 0:email 1:log 2:notes 3:infokiosk 4:research
class HUDLogs extends SqRootScript {
	function playInHUD(logType) {
		// mode check
		if (logType == TYPE_LOG && !AUTO_LOGS) {
			return;
		}
		// working vars
		local questID, hudObj, logTime, logName, logText, logColon, logNewline;
		local logTo = "";
		local logLevelRef = int_ref();
		local logWhichRef = int_ref();
		local logFind = ShockGame.FindLogData(self, logType, logLevelRef, logWhichRef);
		local logLevel = logLevelRef.tointeger();
		local logWhich = logWhichRef.tointeger();
		// only show poup if log/email exists and not already played
		if (logFind == S_OK && ShockGame.TriggerLog(logType, logLevel, logWhich, false)) {
			// display standard notification
			if (logType == TYPE_EMAIL) {
				ShockGame.AddTranslatableText("GetEmail", "misc", 0, 5000);
			}
			// ensure overlay manager object exists
			if (!Object.Named(HUD_CON)) {
				hudObj = Object.Create("Marker");
				Object.SetName(hudObj, HUD_CON)
				Property.Set(hudObj, "Scripts", "Script 0", "zbHudControl");
			}
			else {
				hudObj = Object.Named(HUD_CON);
			}
			// set up data for overlay
			logName = rstrip(logData("Name", logType, logLevel, logWhich));
			if (logType == TYPE_LOG) {
				logText = logData("Text", logType, logLevel, logWhich);
				// check for log having a "To:" recipient in a language-agnostic fashion
				logColon = logText.find(":");
				logNewline = logText.find("\n");
				if (logColon != null && logNewline != null && logNewline > logColon && logColon < 10 && logNewline < 60) {
					logTo = "\n" + logText.slice(0, logNewline);
				}
			}
			_logString = logName + logTo;
			if (_logBitmap) {
				ShockOverlay.FlushBitmap(_logBitmap);
			}
			_logBitmap = ShockOverlay.GetBitmap(logData("Portrait", logType, logLevel, logWhich), "book\\");
			logTime = _logString.find("Chemical Manifest") != null ? 5 : POPUP_TIME;
			// display overlay
			PostMessage(hudObj, "Show",  logTime);
		}
	}

	function logData(logKey, logType, logLevel, logWhich) {
		local prefix = ["Email", "Log"];
		return Data.GetString("level" + pad(logLevel + 1), prefix[logType] + logKey + (logWhich + 1), "");
	}

	function pad(n) {
		return (n < 10 ? "0" : "") + n;
	}
}

// --------------------------------------------------------------------------------
// Picked up audio log
// (overrides ALLOBJS.OSM method)
class LogDiscScript extends HUDLogs {
	function OnFrobWorldEnd() {
		playInHUD(TYPE_LOG);
		Networking.Broadcast("TakeLog", 1);
		// notify engine that log has been picked up (deletes log object)
		if (!ShockGame.UseLog(self, true)) {
			if (HasProperty("QBName") && GetProperty("QBVal") != 0) {
				ShockGame.SetQBHacked(GetProperty("QBName"), GetProperty("QBVal"));
			}
		}
	}

	function OnTakeLog() {
		ShockGame.UseLog(self, false);
	}
}

// --------------------------------------------------------------------------------
// Log trap
// (overrides ALLOBJS.OSM method)
class TrapLog extends HUDLogs {
	function OnTurnOn() {
		playInHUD(TYPE_LOG);
		Link.BroadcastOnAllLinks(self, "TurnOn", "SwitchLink");
		if (HasProperty("QBName") && GetProperty("QBVal") != 0) {
			ShockGame.SetQBHacked(GetProperty("QBName"), GetProperty("QBVal"));
		}
	}
}

// --------------------------------------------------------------------------------
// Email trap
// (overrides ALLOBJS.OSM method)
class TrapEmail extends HUDLogs {
	function OnTurnOn() {
		playInHUD(TYPE_EMAIL);
		Link.BroadcastOnAllLinks(self, "TurnOn", "SwitchLink");
		if (HasProperty("QBName") && GetProperty("QBVal") != 0) {
			ShockGame.SetQBHacked(GetProperty("QBName"), GetProperty("QBVal"));
		}
	}
}

// --------------------------------------------------------------------------------
// Room triggered emails
// (overrides ALLOBJS.OSM method)
// (this is a barely-tested mashup of CoreRoom, BaseRoom, and EmailRoom)
class EmailRoom extends HUDLogs {
	function OnPlayerRoomEnter() {
		local obj = message().MoveObjId;
		if (PlayerEnter(obj)) {
			Link.BroadcastOnAllLinksData(self, "TurnOn", "SwitchLink", obj);
		}
		if (HasProperty("MapLoc")) {
			ShockGame.SetExplored(GetProperty("MapLoc"), 1);
		}
	}

	function OnNetTrigger() {
		SetData("triggered", true);
	}

	function PlayerEnter(obj) {
		if (!GetData("triggered")) {
			playInHUD(TYPE_EMAIL);
			if (HasProperty("QBName") && GetProperty("QBVal") != 0) {
				ShockGame.SetQBHacked(GetProperty("QBName"), GetProperty("QBVal"));
			}
			Networking.Broadcast(self, "NetTrigger", true, null);
			PlayRoomSchemas(obj);
			SetData("triggered", true);
			return true;
		}
		else {
			return false;
		}
	}

	function PlayRoomSchemas(obj) {
		if (HasProperty("Exp")) {
			ShockGame.AddExp(obj, GetProperty("Exp"), true);
			if (HasProperty("ObjSoundName")) {
				Sound.PlaySchemaAmbient(obj, GetProperty("ObjSoundName"));
			}
			else if (GetQVar("ShodanRoom")) {
				Sound.PlaySchemaAmbient(obj, "ExpSHODAN");
			}
			else {
				Sound.PlaySchemaAmbient(obj, "ExpPolito");
			}
		}
		if (HasProperty("UseMsg")) {
			ShockGame.AddTextObjProp(self, "UseMsg", obj, 5000);
		}
	    if (HasProperty("AudioLog")) {
			ShockGame.Play(obj, GetProperty("AudioLog"));
		}
	}
}

// --------------------------------------------------------------------------------
// HUD controller
class zbHudControl extends SqRootScript {
	function destructor() {
		ShockOverlay.RemoveHandler(logOverlay);
		ShockOverlay.FlushBitmap(_logBitmap);
	}

	function OnEndScript() {
		destructor();
	}

	function OnTimer() {
		if (message().name == "Hide") {
			ShockOverlay.RemoveHandler(logOverlay);
		}
	}

	function OnShow() {
		ShockOverlay.AddHandler(logOverlay);
		if (IsDataSet("Timer")) {
			KillTimer(GetData("Timer"));
		}
		SetData("Timer", SetOneShotTimer("Hide", message().data));
	}
}

// --------------------------------------------------------------------------------
// Display log header in HUD
class logOverlayClass extends IShockOverlayHandler {
	function DrawHUD() {
		if (!ShockGame.OverlayOn(kOverlayInv)) {
			ShockOverlay.SetFont(1);
			ShockOverlay.DrawString(_logString, 75, 35);
			ShockOverlay.DrawBitmap(_logBitmap, 10, 35);
		}
	}
}

// This must follow the class definition
logOverlay <- logOverlayClass();
